<?php
session_start();
require_once '../config/db.php';
require_once '../auth/auth.php';
require_once '../helpers/helpers.php';

requireLogin();
$activePage = 'stock_management';

// Handle filters
$start_date = $_GET['start_date'] ?? '';
$end_date = $_GET['end_date'] ?? '';
$product_name = $_GET['product_name'] ?? '';

$where = [];
$params = [];
if ($start_date) {
    $where[] = 'DATE(sa.adjusted_at) >= ?';
    $params[] = $start_date;
}
if ($end_date) {
    $where[] = 'DATE(sa.adjusted_at) <= ?';
    $params[] = $end_date;
}
if ($product_name) {
    $where[] = 'p.name LIKE ?';
    $params[] = "%$product_name%";
}
$where_sql = $where ? 'WHERE ' . implode(' AND ', $where) : '';

// Build WHERE clause for both manual and supplier adjustments
$where_manual = $where ? 'WHERE ' . implode(' AND ', $where) : '';
$where_supplier = [];
if ($start_date) $where_supplier[] = 'DATE(sr.created_at) >= ?';
if ($end_date) $where_supplier[] = 'DATE(sr.created_at) <= ?';
if ($product_name) $where_supplier[] = 'p.name LIKE ?';
$where_supplier_sql = $where_supplier ? 'WHERE ' . implode(' AND ', $where_supplier) : '';

// Combine params for both queries
$params_supplier = [];
if ($start_date) $params_supplier[] = $start_date;
if ($end_date) $params_supplier[] = $end_date;
if ($product_name) $params_supplier[] = "%$product_name%";

$sql = "
    SELECT sa.adjusted_at, p.name as product_name, sa.quantity, sa.reason, u.username as adjusted_by_name, NULL as supplier_id, NULL as supplier_name
    FROM stock_adjustments sa
    JOIN products p ON sa.product_id = p.id
    JOIN users u ON sa.adjusted_by = u.id
    $where_manual
    UNION ALL
    SELECT sr.created_at as adjusted_at, p.name as product_name, sr.quantity, 'From Supplier' as reason, u.username as adjusted_by_name, sr.supplier_id, s.name as supplier_name
    FROM supplier_receipts sr
    JOIN products p ON sr.product_id = p.id
    JOIN users u ON sr.created_by = u.id
    JOIN suppliers s ON sr.supplier_id = s.id
    $where_supplier_sql
    ORDER BY adjusted_at DESC
    LIMIT 20
";
$stmt = $pdo->prepare($sql);
$stmt->execute(array_merge($params, $params_supplier));
$adjustments = $stmt->fetchAll();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Stock Adjustment History - POS System</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;600;700&display=swap" rel="stylesheet">
    <style>
        body { background: #f8fafc; font-family: 'Inter', sans-serif; }
        .card { border-radius: 16px; box-shadow: 0 4px 20px rgba(0,0,0,0.05); border: none; }
        .card-header { background: none; border-bottom: none; padding: 1.5rem 1.5rem 0.5rem; }
        .card-body { padding: 1.5rem; }
        .badge { padding: 0.5em 1em; border-radius: 6px; font-weight: 500; }
    </style>
</head>
<body>
<div class="container-fluid py-4">
    <div class="row">
        <?php include '../includes/sidebar.php'; ?>
        <div class="col-md-10 ms-sm-auto px-md-4">
            <?php include '../includes/topbar.php'; ?>
            <div class="container-fluid mt-4">
                <div class="card mb-4">
                  <div class="card-header d-flex justify-content-between align-items-center">
                    <h5 class="card-title mb-0">Stock Adjustment History</h5>
                    <span class="badge bg-info">Last 20 Adjustments</span>
                  </div>
                  <div class="card-body">
                    <form class="row g-3 mb-3" method="get">
                      <div class="col-md-3">
                        <label class="form-label">Start Date</label>
                        <input type="date" name="start_date" class="form-control" value="<?php echo htmlspecialchars($start_date); ?>">
                      </div>
                      <div class="col-md-3">
                        <label class="form-label">End Date</label>
                        <input type="date" name="end_date" class="form-control" value="<?php echo htmlspecialchars($end_date); ?>">
                      </div>
                      <div class="col-md-4">
                        <label class="form-label">Product Name</label>
                        <input type="text" name="product_name" class="form-control" placeholder="Search by product name..." value="<?php echo htmlspecialchars($product_name); ?>">
                      </div>
                      <div class="col-md-2 d-flex align-items-end">
                        <button type="submit" class="btn btn-primary w-100">Filter</button>
                      </div>
                    </form>
                    <div class="table-responsive">
                      <table class="table table-hover mb-0">
                        <thead>
                          <tr>
                            <th>Date</th>
                            <th>Serial No:</th>
                            <th>Product</th>
                            <th>Quantity</th>
                            <th>Reason</th>
                            <th>Adjusted By</th>
                            <th>From Supplier</th>
                            <th>Manually Adjusted</th>
                          </tr>
                        </thead>
                        <tbody>
                          <?php foreach ($adjustments as $adj): ?>
                          <tr>
                            <td><?php echo date('Y-m-d H:i', strtotime($adj['adjusted_at'])); ?></td>
                            <td>
                              <button type="button" class="btn btn-link btn-sm view-serials-btn" data-product="<?php echo htmlspecialchars($adj['product_name']); ?>" data-date="<?php echo htmlspecialchars($adj['adjusted_at']); ?>">
                                View
                              </button>
                            </td>
                            <td><?php echo htmlspecialchars($adj['product_name']); ?></td>
                            <td>
                              <?php if ($adj['quantity'] > 0): ?>
                                <span class="badge bg-success">+<?php echo $adj['quantity']; ?></span>
                              <?php elseif ($adj['quantity'] < 0): ?>
                                <span class="badge bg-danger"><?php echo $adj['quantity']; ?></span>
                              <?php else: ?>
                                <span class="badge bg-secondary">0</span>
                              <?php endif; ?>
                            </td>
                            <td><?php echo htmlspecialchars($adj['reason']); ?></td>
                            <td><?php echo htmlspecialchars($adj['adjusted_by_name']); ?></td>
                            <td>
                                <?php if (!empty($adj['supplier_name'])): ?>
                                    <?php echo htmlspecialchars($adj['supplier_name']); ?>
                                <?php else: ?>
                                    -
                                <?php endif; ?>
                            </td>
                            <td>
                                <?php if (empty($adj['supplier_id'])): ?>
                                    Yes
                                <?php else: ?>
                                    -
                                <?php endif; ?>
                            </td>
                          </tr>
                          <?php endforeach; ?>
                        </tbody>
                      </table>
                    </div>
                  </div>
                </div>
            </div>
        </div>
    </div>
</div>
<div class="modal fade" id="serialsModal" tabindex="-1" aria-hidden="true">
  <div class="modal-dialog">
    <div class="modal-content">
      <div class="modal-header">
        <h5 class="modal-title">Serial Numbers</h5>
        <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
      </div>
      <div class="modal-body">
        <div class="mb-3">
          <input type="text" class="form-control" id="serialSearchInput" placeholder="Search serial number...">
        </div>
        <div id="serialsList">Loading...</div>
      </div>
    </div>
  </div>
</div>
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
<script>
const filterForm = document.querySelector('form');
const tableBody = document.querySelector('table tbody');

function fetchAdjustments() {
  const formData = new FormData(filterForm);
  const params = new URLSearchParams(formData).toString();
  fetch('fetch_adjustments.php?' + params)
    .then(res => res.text())
    .then(html => {
      tableBody.innerHTML = html;
    });
}

filterForm.addEventListener('input', fetchAdjustments);
filterForm.addEventListener('submit', function(e) {
  e.preventDefault();
  fetchAdjustments();
});

let allSerials = [];

document.querySelectorAll('.view-serials-btn').forEach(btn => {
  btn.addEventListener('click', function() {
    const product = btn.getAttribute('data-product');
    const date = btn.getAttribute('data-date');
    const serialsList = document.getElementById('serialsList');
    const serialSearchInput = document.getElementById('serialSearchInput');
    serialsList.textContent = 'Loading...';
    if (serialSearchInput) serialSearchInput.value = '';
    allSerials = [];
    fetch(`fetch_serials.php?product=${encodeURIComponent(product)}&date=${encodeURIComponent(date)}`)
      .then(res => res.json())
      .then(data => {
        if (data.success && data.serials.length > 0) {
          allSerials = data.serials;
          renderSerials(allSerials);
        } else {
          serialsList.textContent = 'No serial numbers found for this adjustment.';
        }
      })
      .catch(() => {
        serialsList.textContent = 'Error fetching serial numbers.';
      });
    new bootstrap.Modal(document.getElementById('serialsModal')).show();
  });
});

function renderSerials(serials) {
  const serialsList = document.getElementById('serialsList');
  if (serials.length === 0) {
    serialsList.textContent = 'No serial numbers found for this adjustment.';
    return;
  }
  serialsList.innerHTML = '<ul>' + serials.map(s => `<li>${s}</li>`).join('') + '</ul>';
}

document.addEventListener('input', function(e) {
  if (e.target && e.target.id === 'serialSearchInput') {
    const term = e.target.value.trim().toLowerCase();
    const filtered = allSerials.filter(s => s.toLowerCase().includes(term));
    renderSerials(filtered);
  }
});
</script>
</body>
</html> 